//	TorusGamesMouse.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include <math.h>


#ifdef __APPLE__
#pragma mark -
#pragma mark mouse/touch events
#endif

void MouseDown(
	ModelData	*md,
	double		aMouseH,				//	horizontal coordinate (intrinsic units rightward from center, in range [-0.5, +0.5])
	double		aMouseV,				//	 vertical  coordinate (intrinsic units   upward  from center, in range [-0.5, +0.5])
	double		aTimeStamp,				//	in seconds, since some arbitrary "time 0"
	bool		aScrollFlag,			//	Dragging with the shift key requests a scroll (desktop only)
	bool		aMarkFlag,				//	Clicking with the right mouse button or with the control key
										//		(desktop only in both cases)
										//		serves to mark apples in the Apples game.
	bool		aTemporalDoubleClick,	//	Facilitates double-click to exit torus cursor mode (desktop only)
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag)		//	Dragging in response to a flick gesture (iOS only)
{
	md->itsDragBeganTime	= aTimeStamp;
	md->itsPreviousDragTime	= aTimeStamp;

	md->itsCoastingStatus = CoastingNone;	//	If the board is coasting, stop it.

	//	Warning:  On a Mac with a trackpad, two MouseDown() events
	//	may arrive with no intervening MouseUp event.
	//	See "Trackpad behavior" in TorusGamesGraphicsViewMac.m for details.

	if (GameIs3D(md->itsGame))
		MouseDown3D(md, aMouseH, aMouseV, aScrollFlag, aTwoFingerGestureFlag, aFlickGestureFlag);
	else
		MouseDown2D(md, aMouseH, aMouseV, aScrollFlag, aMarkFlag, aTwoFingerGestureFlag, aFlickGestureFlag, aTemporalDoubleClick);
}

//	Technical note:  MouseMove() take parameters (Δh, Δv)
//	because in the desktop version, a 2D torus cursor has
//	no well-defined position in the view, especially not
//	in tiling mode!
void MouseMove(
	ModelData	*md,
	double		aMouseDeltaH,			//	incremental motion, in same coordinate system as in MouseDown()
	double		aMouseDeltaV,			//	incremental motion, in same coordinate system as in MouseDown()
	double		aTimeStamp,				//	in seconds, since some arbitrary "time 0"
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag)		//	Dragging in response to a flick gesture (iOS only)
{
	double	theElapsedTime;

#if TARGET_OS_OSX
	//	macOS generates mouseDragged: events with zero displacement.
	//	I have no idea why.
	//	Ignore them.  As well as being more efficient,
	//	this help avoid jitteriness, for example when dragging the flounder.
	if (aMouseDeltaH == 0.0 && aMouseDeltaV == 0.0)
		return;
#endif

	theElapsedTime			= aTimeStamp - md->itsPreviousDragTime;
	md->itsPreviousDragTime	= aTimeStamp;

	if (GameIs3D(md->itsGame))
		MouseMove3D(md, aMouseDeltaH, aMouseDeltaV, theElapsedTime, aTwoFingerGestureFlag, aFlickGestureFlag);
	else
		MouseMove2D(md, aMouseDeltaH, aMouseDeltaV, theElapsedTime, aTwoFingerGestureFlag, aFlickGestureFlag);
}

void MouseUp(
	ModelData	*md,
	double		aTimeStamp,				//	in seconds, since some arbitrary "time 0"
	bool		aTwoFingerGestureFlag,	//	Dragging in response to a two-finger gesture
	bool		aFlickGestureFlag,		//	Dragging in response to a flick gesture (iOS only)
	bool		aTouchSequenceWasCancelled)
{
	//	Warning:  On a Mac with a trackpad, two MouseUp() events
	//	may arrive with no intervening MouseDown event.
	//	See "Trackpad behavior" in TorusGamesGraphicsViewMac.m for details.
	
	double	theDragDuration;	//	in seconds

	theDragDuration = aTimeStamp - md->itsDragBeganTime;

	if (GameIs3D(md->itsGame))
		MouseUp3D(md, theDragDuration, aTwoFingerGestureFlag, aFlickGestureFlag, aTouchSequenceWasCancelled);
	else
		MouseUp2D(md, theDragDuration, aTwoFingerGestureFlag, aFlickGestureFlag, aTouchSequenceWasCancelled);
}

void MouseGone(ModelData *md)
{
	if ( ! GameIs3D(md->itsGame) )
		MouseGone2D(md);
}

void MouseWheel(
	ModelData	*md,
	double		aDeltaH,
	double		aDeltaV)
{
	if ( ! GameIs3D(md->itsGame) )
		MouseWheel2D(md, aDeltaH, aDeltaV);
}

void MouseCancelled(	//	typically serves as TouchCancelled() on iOS
	ModelData	*md,
	double		aTimeStamp)	//	in seconds, since some arbitrary "time 0"
{
	//	MouseCancelled() may get called under any of the following circumstances:
	//
	//		- when theFlickRecognizer or theTwoFingerRecognizer recognizes its gesture
	//
	//		- when a phone call interrupts the app on an iPhone
	//
	//		- with iOS 9's splitscreen multitasking, when the AppKit
	//			re-interprets an already-underway touch sequence as a motion
	//			of the screen's dividing bar rather than as a drag in the app itself
	//			(however, this tends not to happen in Torus Games because
	//			the game view is always slightly inset within the matte
	//			and thus any affected touch sequences get received by the matte,
	//			not by the game view)
	//

	//	Call MouseUp with aTouchSequenceWasCancelled = true
	MouseUp(md,
			aTimeStamp,
			false,	//	aTwoFingerGestureFlag
			false,	//	aFlickGestureFlag
			true);	//	aTouchSequenceWasCancelled
}


#ifdef __APPLE__
#pragma mark -
#pragma mark coasting
#endif

void CoastingMomentum(
	ModelData	*md,
	double		aTimeInterval)	//	in seconds
{
	if (GameIs3D(md->itsGame))
		CoastingMomentum3D(md, aTimeInterval);
	else
		CoastingMomentum2D(md, aTimeInterval);
}


#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE

#ifdef __APPLE__
#pragma mark -
#pragma mark cursor
#endif

Placement2D GetScaledHandCursorPlacement(
	ModelData	*md)
{
	Placement2D	theHandPlacement;

	//	Copy the current hand placement.
	theHandPlacement = md->its2DHandPlacement;

	//	On the one hand, it would be nice to keep the cursor
	//	a single fixed size at all times.  On the other hand,
	//	it really is too big for the more complicated mazes and apples boards,
	//	so let's make some exceptions there and shrink the cursor
	//	in those cases.
	switch (md->itsGame)
	{
		case Game2DMaze:
			switch (md->itsDifficultyLevel)
			{
				case 0:
				case 1:
					//	Standard cursor size is fine.
					break;
		
				case 2:
					theHandPlacement.itsSizeH *= 0.50;
					theHandPlacement.itsSizeV *= 0.50;
					break;

				case 3:
					theHandPlacement.itsSizeH *= 0.25;
					theHandPlacement.itsSizeV *= 0.25;
					break;
			}
			break;

		case Game2DApples:
			switch (md->itsDifficultyLevel)
			{
				case 0:
				case 1:
					//	Standard cursor size is fine.
					break;
				
				case 2:
					theHandPlacement.itsSizeH *= 0.6;
					theHandPlacement.itsSizeV *= 0.6;
					break;
					
				case 3:
					theHandPlacement.itsSizeH *= 0.3;
					theHandPlacement.itsSizeV *= 0.3;
					break;
			}
			break;

		default:
			//	Standard cursor size is fine.
			break;
	}
	
	return theHandPlacement;
}

#endif	//	TORUS_GAMES_2D_MOUSE_INTERFACE


#ifdef __APPLE__
#pragma mark -
#pragma mark character input
#endif

TitledErrorMessage *CharacterInput(
	ModelData	*md,
	Char16		aCharacter)
{
	TitledErrorMessage	*theErrorMessage;
	
	if (md->itsGameCharacterInput != NULL)
		theErrorMessage = (*md->itsGameCharacterInput)(md, aCharacter);
	else
		theErrorMessage = NULL;

	md->itsChangeCount++;
	
	return theErrorMessage;
}

